% csf.m

% In LINUX, run script from home directory; output will be stored in the same directory

% Original script: Lucas Stam (programming and testing) and Nomdo Jansonius (basic layout and supervision)
% This version is adapted for Linux (octave-psychtoolbox-3) by Nomdo Jansonius
% Latest change: November 5, 2021 by NJ

% (c) Laboratory of Experimental Ophthalmology, University Medical Center Groningen, University of Groningen

% To be cited as: Bierings RAJM, Overkempe T, van Berkel CM, Kuiper M, Jansonius NM (2019) Spatial contrast
% sensitivity from star- to sunlight in healthy subjects and patients with glaucoma. Vision Res 158:31-39.

clear all
close
clc

% INPUT

% Input parameters

m0 = 0.00001;			% Startwaarde modulatiediepte [0.00001; range 0-1]
B=3;				% Grootte pixelblokken [3 pixels; range 1-4]
f=8;				% Beeldfrequentie in Hz [8 Hz; range 2-10]
L0=150;				% Indexwaarde gemiddelde luminantie [150; range 50-150]
N=6;				% Aantal meetpunten [6; min 5]
mdot=0.3;			% Logaritmische verandering modulatiediepte [0.3 log/s; max 0.5]
d=3;				% Afstand proefpersoon in meters [3 m; max 8]
W=148;				% Lijnlengte in millimeters [148 mm; min 100]

disp('current presumed testing distance (m):'); d

casenr = input('Case number of participant [9999]:                        ');
if isempty(casenr)
  casenr = 9999; % Default answer is 9999
end

cpd    = input('Spatial frequency [3 cpd]:                                ');
if isempty(cpd) 
  cpd = 3; % Default value in case no input is given is 3 cpd
elseif cpd > 16
  disp('Spatial frequency must be in between 1 and 16 cpd');
  clear cpd
cpd    = input('Spatial frequency [3 cpd]:                                ');
if isempty(cpd)
  cpd = 3;
end
end

fm     = input('temporal frequency [0 Hz]:                                '); 
if isempty(fm) 
  fm = 0; % Default value in case no input is given is 0 Hz
elseif fm > f/2
  disp('Maximum temporal frequency equals refresh rate f/2'); f
  clear fm
fm     = input('temporal frequency [0 Hz]:                                ');
if isempty(fm)
  fm = 0;
end
end

disp('current calibration line length (mm):'); W
reply  = input('Do you want to check line length? Y = 1, N = 2 [N]:       ');
if isempty(reply)
  reply = 2; % Default answer is no
end

% If one answered yes to the above question, below lines summon a line on the screen
% of 500 pixels length. Measurement of this line provides the pixel density.

if reply == 1;
  screenid=0;
  R0 = monitorfunctie(100);

  win = Screen('OpenWindow', screenid, [R0 R0 R0]);
  [w, h] = Screen('WindowSize', win);
  ifi = Screen('GetFlipInterval', win);
  vbl=Screen('Flip', win);

  button =0;

  while ~button(1)
    % Query mouse:
    [xm, ym, button] = GetMouse;
    Screen('FillRect',win,[0 0 0],[(w/2)-250 (h/2)-3 (w/2)+250 (h/2)+3]);
    Screen('TextSize',win, 20);
    Screen('DrawText',win,'Measure the length of the line below',100,100,[0 0 0]);
    Screen('DrawText',win,'If ready, press left mouse button',100,50,[0 0 0]);
    vbl=Screen('Flip',win,vbl+ifi/2);
  end

  Screen('Close', win);

end

% PARAMETERS AND INITIAL VALUES

% No bright white welcome screen of PSYCHTOOLBOX
Screen('Preference', 'VisualDebuglevel', 3);

% Which screen to apply the stimulus on
screenid = 0;

% Open window with black background color
win = Screen('OpenWindow', screenid, [0,0,0]);
[w, h] = Screen('WindowSize', win);
ifi = Screen('GetFlipInterval', win);			% ifi = 1/60 = 0.0167
vbl = Screen('Flip', win);				% vbl = 3163

% Conversion of variables to pixels instead of arc seconds

realifi = 1/f; % This is a fixed frame interval which has to be set large
% enough to make sure that it exceeds the computation time per frame.

den = (500*1000/W); % Calculating pixel density of monitor; pixels/meter
pp = 2*pi*d/(360*cpd)*den; % period length of stimulus in pixels

button = [0 0]; % default button 
M = 0; % initial parameter for button(1)
TIM = 0; % initial parameter for timer
del = 0; % initial parameter for delay

E = zeros(1,2*N); % This vector will store the values of the amplitudes
DEL = zeros(1,2*N); % This vector will store the values of the time delays

m = m0; % This is the initial modulation depth

% Number of dead pixels at the right side of the screen due to the block nature of the stimulus
dpr = mod(w,N); % dead pixels right
dpb = mod(h,N); % dead pixels bottom

% Create matrix dimensions [h,w]. This matrix represents the x-coordinates of the individual
% pixels of the screen, minus the dead pixels on the right side of the screen.

x = 1:(w-dpr);
t = 0; % The initial value of the timer used to calculate the temporal modulation of the spatial pattern
C = 0; %number of missed frames

% STIMULUS

while ~ button(2) % The experiment is stopped by pressing the middle button
 
tic
    % Query mouse:
    [xm, ym, button] = GetMouse;

    % Pressing the right mouse button pauses the animation
    if button(3) == 0
     
      % Below loop saves the amplitude of oscillation when the button is pressed in a vector E
        
      if (button(1)+M) == 1;
        Fi = find(E==0);
        E(Fi(1)) = m; % Add a value of m to the vector
        M = button(1); % The parameter M1 is equal to the current value of the button(1)
        del = ceil(f*rand); % Calculate a random delay between 1 and f frames
        Fi = find(DEL==0);
        DEL(Fi(1)) = del; % Add a value of del to the vector   
        TIM = 0; % timer, counts the amount of frames refreshed since last time the button was pressed/released
      end   
        
      if nnz(E) == 2*N % Check whether one obtained enough data
        button(2) = 1; % Automatically end the stimulus
      end

      LV = L0*(1+m*sin(2*pi*(x)/pp)*cos(2*pi*fm*t)); % make the luminance index vector
      t = t+realifi; % produce the next value of the timer 
      RV = monitorfunctie(LV); % convert to RGB vector 

      % Command below creates a vector with triplets containing the average
      % value of the corresponding elements in the vector RV
      MR = reshape( ones(B)/B*reshape(RV,B,[]) , 1,[]); % MR means mean RGB values
      MR = MR(ones(1,B), :); % Vertically concatenate the vector B times with itself
      % Takes negligible amount of time
      MRF = floor((MR-floor(MR))*B^2);
      % Takes negligible amount of time

% Lines below create a B by (w-dpr) matrix which contains blocks of B^2 pixels.
% Each of the elements is an integer RGB value which will be sent to the
% monitor. Each block holds a specific number of elements with a higher RGB
% number to mimic the average desired RGB value of the entire block

% Example for blocks of width 3. If the script calculates that the average
% of the block RGB is 140.37, the script will round this down to the nearest
% fraction of nine, that would be 140.3333. The block will than contain six
% pixels with RGB value 140, and 3 pixels with RGB value of 141. The pixels
% with a higher RGB value are distributed along the block in an orderly fashion,
% starting in the upper left corner and ending in the lower right corner.

      DR = reshape(MRF,B^2,[]); % Creates a matrix which will be used in calculation only
      A = zeros(B^2, columns(DR));
      for y = 1:columns(DR)
        A(1:DR(1,y),y) = 1;
      end
      Z = reshape(A,B,length(MR));
      RVB = floor(MR)+Z;
      RVB = [RVB zeros(B,dpr)]; % add some zeros to the matrix to account
                                % for the dead pixels on the right side of the screen

      % RVB refers to a R(GB) vector with a vertical dimension B (pixel block size)
      % We now have a B x w vector which has to be extended in the vertical direction to cover the entire screen. 
 
      RM = zeros(h,w); % RM stands for RGB matrix. The matrix RM will be used to drive the monitor.

      % Extended the vector RVB downwards to construct a matrix RM. RM stands for RGB matrix. 
      % The matrix RM will be used to drive the monitor. 
      for z = 0:floor(h/B-1)
        for z1 = 1:B
          RM(B*z+z1,:) = RVB(z1,:);
        end
      end
       
      tex = Screen('MakeTexture',win,RM,specialFlags=1); % makes texture from matrix: TIME CONSUMING!
      Screen('DrawTexture',win,tex); % draws the texture on the screen
      Screen('Close',tex); % close down the proxy window created by 'MakeTexture'
    
      % Below lines detect how many times the screen has refreshed since the last time the button was pressed

      TIM = TIM+1;

      % During the delay no changes are made to the modulation depth
      if TIM > del
        m = m*10^((1-2*M)*mdot*realifi); % Calculation of new modulation depth; the new sign of the button is not yet taken into account
      end   
      vbl = Screen('Flip', win); % The screen is refreshed, above Screen() commands are made visible on the monitor 
    end

    while toc < realifi % This make sure the screen is refreshed at the artificially set refresh rate
  end
end

Screen('Close', win);

% DATA ANALYSIS

% Rather than taking the linear values of the contrasts at the extremes
% (reverals of the von Bekesy tracking), we take the logarithm of it.
Ex = log10(E);

% Determining the threshold from the Von Bekesy data. In this method one
% removes the first and second data point, as well as the most and least
% extreme minimum and maximum from analysis (in total 6 points). 

% find highest maximum Ex other than first two points:
k = 2*find(Ex(3:2:length(Ex)) == max(Ex(3:2:length(Ex))))+1; 
if length(k) > 1 % Just one point is removed in case two extremes are equal
  k = k(end);
end
K1 = [1 2 k]; 
% find lowest maximum Ex other than first two points:
k = 2*find(Ex(3:2:length(Ex)) == min(Ex(3:2:length(Ex))))+1;
if length(k) > 1 % Just one point is removed in case two extremes are equal
  k = k(end);
end
K1 = [K1 k];
% find lowest minimum Ex other than first two points:
k = 2*find(Ex(4:2:length(Ex)) == min(Ex(4:2:length(Ex))))+2; 
if length(k) > 1 % Just one point is removed in case two extremes are equal
  k = k(end);
end
K1 = [K1 k];
% find highest minimum Ex other than first two points:
k = 2*find(Ex(4:2:length(Ex)) == max(Ex(4:2:length(Ex))))+2; 
if length(k) > 1 % Just one point is removed in case two extremes are equal
  k = k(end);
end

% The vector K1 contains the indices of the data points which need to be removed for analysis
K1 = [K1 k]; 

K = 1:1:2*N;
K(K1) = []; % This vector contains all integers of the data points in Ex which will be used for data analysis
K1max = K(mod(K,2) == 1); % integers of remaining maxima in the vector Ex
K1min = K(mod(K,2) == 0); % integers of remaining minima in the vector Ex

% Amplitude A1 is calculated by taking the average of all remaining data points of Ex.
% The error in A1 is calculated by the quadratic sum of the error in the minima of Ex and the error in the maxima of E1.

Mod = sum(Ex(K))/length(Ex(K)); % Calculation of log of modulation (contrast threshold)
SD = 0.5*sqrt((std(Ex(K1max)))^2+(std(Ex(K1min)))^2); % Calculation of SD
logCS = -1*Mod; % log of contrast sensitivity

% PLOT RAW RESULTS

DELT = (1/f)*DEL; % Delay expressed in seconds rather than in frames

EX = zeros(1,4*N);
EX(1) = log10(m0);
EX(2:2:4*N) = Ex;
EX(3:2:4*N-1) = Ex(1:2*N-1);

% Just to summarize:
% Ex contains 2*N extreme values
% EX contains 4*N points: the starting point, all the extreme points, all the extreme points + delay

% Create a string of the times that corresponds to the values of the modulation depth provided in vector EX.
% This vector will be used to plot the modulation as a function of time.
T = zeros(1,4*N);
for J = 1:2*N;
  T(2*J) = abs((EX(2*J)-EX(2*J-1))/mdot)+T(2*J-1);
  T(2*J+1) = T(2*J)+DELT(J);
end

T(4*N+1) = []; % remove last delay since it doesn't do anything (script is stopped after last data point is acquired). 

% K1 contains indices of removed values
% K contains indices of retained values
figure(20,'Visible','off') % creating a figure without showing this on the screen.

figurename = ['log contrast as a function of time'];
set(gcf,'name',figurename)
plot(T,EX);
hold on
plot(T(2*K1),EX(2*K1),'*') % Mark removed values with asterix
% K = [K1min K1max]+1;
plot(T(2*K),EX(2*K),'o') % Mark retained values with dot
hold off

xlabel('time (s)')
ylabel('log contrast')
title('log contrast as a function of time')

% WRITING DATA TO DISK

casenrstr = num2str(casenr);

filename = strcat('CSF-graph-',casenrstr,'.jpg'); % creating a filename for the graph
print(filename); % saves the graph to a .jpg file 

filename = strcat('CSF-data-',casenrstr,'.csv'); % creating a filename for the data file

% Everytime the script is run, a line of information is added that contains: 
% case number of participant, date and time of measurement, and summary results and stimulus parameters:
% logCS, spatial frequency, temporal frequency, index of mean luminance,
% distance between monitor and subject, rate of change of contrast, initial contrast, and raw data of reversals
dlmwrite(filename,[casenr,datevec(now),logCS,cpd,fm,L0,d,mdot,m0,Ex],'-append','delimiter',',');

%% DISPLAY THE MAIN RESULT IN OCTAVE/MATLAB

disp('log contrast sensitivity is'), disp(logCS);
